﻿using iText.Kernel.Pdf;
using NVCC.Models;
using NVCC.WebUI.Models;
using System;
using System.Collections.Generic;
using System.Configuration;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Web;
using System.Xml;
using System.Xml.Linq;
using System.Xml.Serialization;

namespace NVCC.WebUI.Infrastructure
{
    public class ImageService : IImageService
    {
        private static readonly bool UseTestVistAImagingServer = Convert.ToBoolean(ConfigurationManager.AppSettings["UseTestVistAImagingServer"]);
        private static readonly bool VistAImaginglogging = ConfigurationManager.AppSettings["VistAImagesLoggingEachAPICallDetailedEnabled"] == "true";
        private static readonly bool VistAImagingTotalTimelogging = ConfigurationManager.AppSettings["VistAImagesAPICallsLoggingSimpleEnabled"] == "true";

        private Site GetClosestServerForCurrentUser(short sta6a)
        {
            Stopwatch totalClosestServerTime = new Stopwatch();
            if (VistAImaginglogging)
            {
                totalClosestServerTime = Stopwatch.StartNew();
            }
            string url = ConfigurationManager.AppSettings["ImagingLoadBalancerServerURL"] + sta6a;
            //load blalancer for production & staging environmentonly
            string env = "prod";
            Stopwatch sw = new Stopwatch();
            if (VistAImaginglogging)
            {
                sw = Stopwatch.StartNew();
            }
            string targetServers = Utilities.CustomWebRequest.ExecuteRequest(url, env);
            if (VistAImaginglogging)
            {
                sw.Stop();
                Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Elapsed time:" + sw.Elapsed + ";Closest server url:" + url));
            }
            if (string.IsNullOrEmpty(targetServers))
            {
                //If the first call failed try a second time
                targetServers = Utilities.CustomWebRequest.ExecuteRequest(url, env);
                if (string.IsNullOrEmpty(targetServers))
                    return null;
            }

            if (VistAImagingTotalTimelogging)
            {
                totalClosestServerTime.Stop();
                Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Elapsed time to retrieve VIX server: " + totalClosestServerTime.Elapsed));
            }
            return MappingClosestServerServiceResultSet(targetServers);
        }

        public void GetImagesMetaData(PatientProfileViewModel patientProfileViewModel, bool includeDetailedIndividualStudyData = false)
        {
            if (!UseTestVistAImagingServer || HttpContext.Current.Request.Url.AbsoluteUri.Contains("staging") || HttpContext.Current.Request.Url.AbsoluteUri.Contains("refdoc.va.gov"))
            {
                GetProdImagesMetaData(patientProfileViewModel, includeDetailedIndividualStudyData);
            }
            else
            {
                GetTestImagesMetaData(patientProfileViewModel, includeDetailedIndividualStudyData);
            }
        }

        public Site MappingClosestServerServiceResultSet(string xmlString)
        {
            Site site = new Site();
            string result = string.Empty;

            // If loadbalancer service has a VIXS server for current station otherwise throw an error
            if (xmlString.Contains("VIXS") || xmlString.Contains("vixs"))
            {
                result = xmlString.Replace(" <", "<");
            }
            else
            {
                Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Load balancer doesn't have a VIXS server for this station"));
                return null;
            }

            //Remove Whitespace from XML Tags
            result = result.Replace(" <", "<");

            //Convert XML to an object 
            XmlSerializer serializer = new XmlSerializer(typeof(Site));
            StringReader stringReader = new StringReader(result);
            site = (Site)serializer.Deserialize(stringReader);

            return site;
        }

        public void MappingStudyServiceResultSet(string xmlString,  string rootUrl, PatientProfileViewModel patientProfileViewModel, string env, string authToken, bool includeDetailedIndividualStudyData = false)
        {
            List<Study> latestStudies = new List<Study>();
            Stopwatch totalstudiesTime = new Stopwatch();
            if (VistAImagingTotalTimelogging)
            {
                totalstudiesTime = Stopwatch.StartNew();
            }
            if (patientProfileViewModel.PatientProfile.ImagingStudiesResultSet == null || !string.IsNullOrWhiteSpace(patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Errors))
            {
                if (string.IsNullOrEmpty(xmlString))
                {
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Load balancer service is broken"));
                    return;
                }

            //Remove Whitespace from XML Tags
            string result = xmlString.Replace("> ", ">");
            result = result.Replace(" <", "<");

                //Convert XML to an object 
                XmlSerializer serializer = new XmlSerializer(typeof(StudiesResult));
                StringReader stringReader = new StringReader(result);
                patientProfileViewModel.PatientProfile.ImagingStudiesResultSet = (StudiesResult)serializer.Deserialize(stringReader);

                DateTime sixmonthsAgo = DateTime.Today.AddMonths(-6);
                
                foreach (var study in patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study)
                {
                    if (Convert.ToDateTime(study.ProcedureDate).CompareTo(sixmonthsAgo) > 0)
                    {
                        latestStudies.Add(study);
                    }
                }
            }
            else
            {
                latestStudies = patientProfileViewModel.PatientProfile.ImagingStudiesResultSet?.Studies?.Study;
            }

            if (includeDetailedIndividualStudyData && latestStudies != null && latestStudies.Count > 0)
            {
            var url = ConfigurationManager.AppSettings["SpecificStudyRelativeUrl"];

                // Retrieve all images for each study
                for (int i = 0; i < latestStudies.Count; i++)
                {
                    if (latestStudies[i].ImageCount <= 1 || !patientProfileViewModel.VistAImagingStudiesSelected[i] ||
                        latestStudies[i].Serieses != null) continue;
                    Stopwatch sw = new Stopwatch();
                    if (VistAImaginglogging)
                    {
                        sw = Stopwatch.StartNew();
                    }
                    var xmlString4Study = Task.Run(async () => await Utilities.CustomWebRequest.ExecuteStudyRequest(rootUrl, url + latestStudies[i].StudyId.Trim(), env, authToken, patientProfileViewModel));
                    if (string.IsNullOrEmpty(xmlString4Study.Result))
                    {
                        //If the call failed then try again a second time
                        xmlString4Study = Task.Run(async () => await Utilities.CustomWebRequest.ExecuteStudyRequest(rootUrl, url + latestStudies[i].StudyId.Trim(), env, authToken, patientProfileViewModel));
                    }
                    if (VistAImaginglogging)
                    {
                        sw.Stop();
                        Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Elapsed time:" + sw.Elapsed + ";study url:" + url));
                    }
                    if (!string.IsNullOrWhiteSpace(xmlString4Study.Result))
                    {
                        latestStudies[i] = MappingImagesForSpecificStudy(xmlString4Study.Result);
                    }
                }
            }

            if (patientProfileViewModel.PatientProfile.ImagingStudiesResultSet?.Studies != null)
            {
                patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study = latestStudies;
            }
            if (VistAImagingTotalTimelogging)
            {
                totalstudiesTime.Stop();
                Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Total elapsed time to retrieve all studies: " + totalstudiesTime.Elapsed));
            }
        }

        public Study MappingImagesForSpecificStudy(string xmlString)
        {
            Study results = new Study();
            if (string.IsNullOrEmpty(xmlString))
            {
                Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("xmlString was null or empty for specific study"));
                return results;
            }

            //Remove Whitespace from XML Tags
            string result = xmlString.Replace("> ", ">");
            result = result.Replace(" <", "<");

            //Convert XML to an object 
            XmlSerializer serializer = new XmlSerializer(typeof(Study));
            StringReader stringReader = new StringReader(result);
            results = (Study)serializer.Deserialize(stringReader);

            return results;
        }



        public void GetProdImagesMetaData(PatientProfileViewModel patientProfileViewModel, bool includeDetailedIndividualStudyData = false)
        {
            //These 4 lines of code below is to handle the case when a user doesn't have VistA access
            StudiesResult resultSet = new StudiesResult();
            Studies studies = new Studies();
            studies.Study = new List<Study>();
            resultSet.Studies = studies;

            string xmlString = string.Empty;
            string rootUrl = string.Empty;
            string url = string.Empty;
            string env = "prod";

            Stopwatch totalMetaDataTime = new Stopwatch();
            if (VistAImagingTotalTimelogging)
            {
                totalMetaDataTime = Stopwatch.StartNew();
            }

            string authToken = HttpContextManager.Current.Session["ImageAuthToken"].ToString();

            if (patientProfileViewModel.PatientProfile.VistAImagingSite == null)
            {
                patientProfileViewModel.PatientProfile.VistAImagingSite = GetClosestServerForCurrentUser(patientProfileViewModel.PatientProfile.UserInfo
                    .CurrentDefaultFacility);
            }

            if (patientProfileViewModel.PatientProfile.VistAImagingSite == null)
            {
                Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Load balancer service is broken"));
            }
            else
            {
                Connection connection = patientProfileViewModel.PatientProfile.VistAImagingSite.Connections.Connection.Find(x => x.Protocol == "VIXS");
                rootUrl = "https://" + connection.Server+"/";
                url = rootUrl + "Study/restservices/studies/site/icn/" + patientProfileViewModel.PatientProfile.UserInfo.CurrentDefaultFacility + "/" + patientProfileViewModel.PatientIcn;
            }

            if (patientProfileViewModel.PatientProfile.ImagingStudiesResultSet == null ||
                !string.IsNullOrWhiteSpace(patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Errors))
            {

                if (!string.IsNullOrEmpty(url))
                {
                    Stopwatch sw = new Stopwatch();
                    if (VistAImaginglogging)
                    {
                        sw = Stopwatch.StartNew();
                    }
                    xmlString = Utilities.CustomWebRequest.ExecuteRequest(url, env, authToken, patientProfileViewModel);

                    if (string.IsNullOrEmpty(xmlString))
                    {
                        //If the first call failed then try a second time
                        xmlString = Utilities.CustomWebRequest.ExecuteRequest(url, env, authToken,
                            patientProfileViewModel);
                        if (string.IsNullOrEmpty(xmlString))
                            return;
                        //Not sure if maybe here we could set resultSet.Errors with something and then set patientProfileViewModel.ImagingStudiesResultSet = resultSet;
                        // What is supposed to happen when we can't get a response from VistA Imaging?
                    }
                    if (VistAImaginglogging)
                    {
                        sw.Stop();
                        string relativeUrl = "Study/restservices/studies/site/icn/" + patientProfileViewModel.PatientProfile.UserInfo.CurrentDefaultFacility;
                        Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Elapsed time: " + sw.Elapsed + "; metadata url:" + relativeUrl));
                    }
                }

                if (xmlString == "Forbidden")
                {
                    resultSet.Errors = "Forbidden";
                    patientProfileViewModel.PatientProfile.ImagingStudiesResultSet = resultSet;
                    return;
                }

                if (xmlString == "NotFound")
                {
                    resultSet.Errors = "NotFound";
                    patientProfileViewModel.PatientProfile.ImagingStudiesResultSet = resultSet;
                    return;
                }

                if (xmlString == "SIGNONS")
                {
                    resultSet.Errors = "SIGNONS";
                    patientProfileViewModel.PatientProfile.ImagingStudiesResultSet = resultSet;
                    return;
                }
            }

            if (VistAImagingTotalTimelogging)
            {
                totalMetaDataTime.Stop();
                Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Total elapsed time to retrieve VistA prod metadata: " + totalMetaDataTime.Elapsed));
            }
            MappingStudyServiceResultSet(xmlString, rootUrl, patientProfileViewModel, env, authToken, includeDetailedIndividualStudyData);
        }

        public void GetTestImagesMetaData(PatientProfileViewModel patientProfileViewModel, bool includeDetailedIndividualStudyData = false)
        {
            //These 4 lines of code below is to handle the case when a user doesn't have VistA access
            StudiesResult resultSet = new StudiesResult();
            Studies studies = new Studies();
            studies.Study = new List<Study>();
            resultSet.Studies = studies;

            Stopwatch totalMetaDataTime = new Stopwatch();
            if (VistAImagingTotalTimelogging)
            {
                totalMetaDataTime = Stopwatch.StartNew();
            }

            string xmlString = string.Empty;
            string testIcn = string.Empty;
            string env = "test";
            string authToken = ConfigurationManager.AppSettings["ImageAuthToken4Test"];
            string rootUrl = ConfigurationManager.AppSettings["ImagingTestServerURL"];

            string url = rootUrl + ConfigurationManager.AppSettings["StudyServiceRelativeUrl"];
            if (HttpContextManager.Current.Session["ICN"] != null)
            {
                url = url + HttpContextManager.Current.Session["ICN"];
            }
            else
            {
                testIcn = new ImageService().getRandomICN();
                url = url + testIcn;
                HttpContextManager.Current.Session["ICN"] = testIcn;
            }

            if (patientProfileViewModel.PatientProfile.ImagingStudiesResultSet == null ||
                !string.IsNullOrWhiteSpace(patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Errors))
            {
                Stopwatch sw = new Stopwatch();
                if (VistAImaginglogging)
                {
                    sw = Stopwatch.StartNew();
                }
                xmlString = Utilities.CustomWebRequest.ExecuteRequest(url, env, authToken, patientProfileViewModel);
                if (VistAImaginglogging)
                {
                    sw.Stop();
                    string relativeUrl = ConfigurationManager.AppSettings["StudyServiceRelativeUrl"];
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Elapsed time:" + sw.Elapsed + ";test metadata url:" + relativeUrl));
                }

                if (string.IsNullOrEmpty(xmlString))
                {
                    //If the first call failed then try a second time
                    xmlString = Utilities.CustomWebRequest.ExecuteRequest(url, env, authToken, patientProfileViewModel);
                    if (string.IsNullOrEmpty(xmlString))
                        return;
                }

                if (xmlString == "Forbidden")
                {
                    resultSet.Errors = "Forbidden";
                    patientProfileViewModel.PatientProfile.ImagingStudiesResultSet = resultSet;
                    return;
                }

                if (xmlString == "NotFound")
                {
                    resultSet.Errors = "NotFound";
                    patientProfileViewModel.PatientProfile.ImagingStudiesResultSet = resultSet;
                    return;
                }

                if (xmlString == "SIGNONS")
                {
                    resultSet.Errors = "SIGNONS";
                    patientProfileViewModel.PatientProfile.ImagingStudiesResultSet = resultSet;
                    return;
                }
            }
            if (VistAImagingTotalTimelogging)
            {
                totalMetaDataTime.Stop();
                Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Total elapsed time to retrieve VistA test metadata: " + totalMetaDataTime.Elapsed));
            }
            MappingStudyServiceResultSet(xmlString, rootUrl, patientProfileViewModel, env, authToken, includeDetailedIndividualStudyData);
        }

        private static readonly bool RetrieveDiagnosticVistAImages = Convert.ToBoolean(ConfigurationManager.AppSettings["RetrieveDiagnosticVistAImages"]);
        private static readonly bool RetrieveReferenceVistAImages = Convert.ToBoolean(ConfigurationManager.AppSettings["RetrieveReferenceVistAImages"]);
        public void SetupVistAImages(PatientProfileViewModel patientProfileViewModel)
        {
            if (patientProfileViewModel.PatientProfile.ImagingStudiesResultSet?.Studies?.Study == null 
                || !patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study.Any() 
                || patientProfileViewModel.VistAImagingStudiesSelected == null 
                || !patientProfileViewModel.VistAImagingStudiesSelected.Any(v => v))
            {
                return;
            }

            Stopwatch totalTime = new Stopwatch();
            if (VistAImagingTotalTimelogging)
            {
                totalTime = Stopwatch.StartNew();
            }

            string url = string.Empty;
            string env = string.Empty;
            string authToken = string.Empty;
            Stopwatch sw = new Stopwatch();
            Stopwatch sw2 = new Stopwatch();
            if (!UseTestVistAImagingServer || HttpContext.Current.Request.Url.AbsoluteUri.Contains("staging") || HttpContext.Current.Request.Url.AbsoluteUri.Contains("refdoc.va.gov"))
            {
                if (patientProfileViewModel.PatientProfile.VistAImagingSite == null)
                {
                    patientProfileViewModel.PatientProfile.VistAImagingSite = GetClosestServerForCurrentUser(patientProfileViewModel.PatientProfile.UserInfo
                        .CurrentDefaultFacility);
                }

                if (patientProfileViewModel.PatientProfile.VistAImagingSite == null)
                {
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Load balancer service is broken"));
                }
                else
                {
                    Connection connection = patientProfileViewModel.PatientProfile.VistAImagingSite.Connections.Connection.Find(x => x.Protocol == "VIXS");
                    url = "https://" + connection.Server;
                    env = "prod";
                    authToken = HttpContextManager.Current.Session["ImageAuthToken"].ToString();
                }
            }
            else
            {
                url = ConfigurationManager.AppSettings["ImagingTestServerURL"];
                env = "test";
                authToken = ConfigurationManager.AppSettings["ImageAuthToken4Test"];
            }

            if (!string.IsNullOrEmpty(url))
            {
                if (patientProfileViewModel.PatientProfile.VistAImagingStudies == null)
                {
                    patientProfileViewModel.PatientProfile.VistAImagingStudies = new List<VistAImagingStudy>();
                }

                for (int i = 0; i < patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study.Count; i++)
                {
                    if (i< patientProfileViewModel.VistAImagingStudiesSelected.Count 
                        && patientProfileViewModel.VistAImagingStudiesSelected[i] 
                        && patientProfileViewModel.PatientProfile.VistAImagingStudies.All(s => s.StudyId != patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study[i].StudyId))
                    {
                        var vistAImagingStudy = new VistAImagingStudy();
                        vistAImagingStudy.StudyId = patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study[i].StudyId;
                        vistAImagingStudy.ShortDescription = patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study[i].Description;
                        vistAImagingStudy.Title = patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study[i].NoteTitle;
                        vistAImagingStudy.Specialty = patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study[i].SpecialtyDescription;
                        vistAImagingStudy.Procedure = patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study[i].ProcedureDescription;
                        vistAImagingStudy.ProcedureDateTime = Convert.ToDateTime(patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study[i].ProcedureDate);
                        vistAImagingStudy.TotalImageCount = patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study[i].ImageCount;


                        vistAImagingStudy.Files = new List<VistAImagingFile>();

                        if (patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study[i].ImageCount > 1
                            && patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study[i].Serieses != null
                            && patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study[i].Serieses.Series != null
                            && patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study[i].Serieses.Series.Any(s => s.Images?.Image != null && s.Images.Image.Any()))
                        {
                            foreach (var series in patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study[i].Serieses.Series)
                            {

                                if (series.Images?.Image != null)
                                {

                                    foreach (var image in series.Images.Image)
                                    {
                                        bool isPdf = image.ImageType.ToLower().Contains("pdf");
                                        var diagnosticImageUri = GetRestrictedFileTypeUri(image.DiagnosticImageUri, isPdf);
                                        var referenceImageUri = GetRestrictedFileTypeUri(image.ReferenceImageUri, isPdf);
                                        try
                                        {
                                            VistAImagingFile vistAFile = null;
                                            if (RetrieveDiagnosticVistAImages)
                                            {
                                                if (VistAImaginglogging)
                                                {
                                                    sw = Stopwatch.StartNew();
                                                }
                                                Task<VistAImagingFile> diagnosticImageTask = Task.Run(async () =>
                                                    await Utilities.CustomWebRequest.ExecuteImageRequest(url,
                                                        diagnosticImageUri, env, authToken, patientProfileViewModel));

                                                vistAFile = diagnosticImageTask.Result;
                                                if (VistAImaginglogging)
                                                {
                                                    sw.Stop();
                                                    Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Elapsed time: " + sw.Elapsed + "; image url:" + diagnosticImageUri));
                                                }
                                            }

                                            if (RetrieveReferenceVistAImages && (vistAFile == null || !IsValidImageOrPdf(vistAFile)))
                                            {
                                                if (VistAImaginglogging)
                                                {
                                                    sw2 = Stopwatch.StartNew();
                                                }
                                                Task<VistAImagingFile> referenceImageTask = Task.Run(async () => await Utilities.CustomWebRequest.ExecuteImageRequest(url, referenceImageUri, env, authToken, patientProfileViewModel));

                                                vistAFile = referenceImageTask.Result;
                                                if (VistAImaginglogging)
                                                {
                                                    sw2.Stop();
                                                    Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Elapsed time: " + sw2.Elapsed + "; image url:" + referenceImageUri));
                                                }
                                            }
                                            
                                            if (vistAFile != null && IsValidImageOrPdf(vistAFile))
                                            {
                                                vistAImagingStudy.Files.Add(vistAFile);
                                            }
                                        }
                                        catch (Exception e)
                                        {
                                            vistAImagingStudy.ErrorMessage = "This VistA image could not be converted into a format that can be included in a PDF. It WILL NOT be included in the RefDoc package.";
                                            vistAImagingStudy.ExceptionMessage = e.Message;
                                        }
                                    }
                                }
                            }

                        }
                        else
                        {
                            var firstImage = patientProfileViewModel.PatientProfile.ImagingStudiesResultSet.Studies.Study[i].FirstImage;
                            if (firstImage != null)
                            {
                                bool isPdf = firstImage.ImageType.ToLower().Contains("pdf");
                                var diagnosticImageUri = GetRestrictedFileTypeUri(firstImage.DiagnosticImageUri, isPdf);
                                var referenceImageUri = GetRestrictedFileTypeUri(firstImage.ReferenceImageUri, isPdf);
                                try
                                {
                                    VistAImagingFile vistAFile = null;
                                    if (RetrieveDiagnosticVistAImages)
                                    {
                                        if (VistAImaginglogging)
                                        {
                                            sw = Stopwatch.StartNew();
                                        }
                                        Task<VistAImagingFile> diagnosticImageTask = Task.Run(async () =>
                                        await Utilities.CustomWebRequest.ExecuteImageRequest(url, diagnosticImageUri,
                                            env, authToken, patientProfileViewModel));

                                    vistAFile = diagnosticImageTask.Result;
                                        if (VistAImaginglogging)
                                        {
                                            sw.Stop();
                                            Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Elapsed time: " + sw.Elapsed + "; image url:" + diagnosticImageUri));
                                        }
                                    }

                                    if (RetrieveReferenceVistAImages && (vistAFile == null || !IsValidImageOrPdf(vistAFile)))
                                    {
                                        if (VistAImaginglogging)
                                        {
                                            sw2 = Stopwatch.StartNew();
                                        }
                                        Task<VistAImagingFile> referenceImageTask = Task.Run(async () =>
                                            await Utilities.CustomWebRequest.ExecuteImageRequest(url, referenceImageUri,
                                                env, authToken, patientProfileViewModel));

                                        vistAFile = referenceImageTask.Result;
                                        if (VistAImaginglogging)
                                        {
                                            sw2.Stop();
                                            Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Elapsed time: " + sw2.Elapsed + "; image url:" + referenceImageUri));
                                        }
                                    }


                                    if (vistAFile != null && IsValidImageOrPdf(vistAFile))
                                    {
                                        vistAImagingStudy.Files.Add(vistAFile);
                                    }
                                }
                                catch (Exception e)
                                {
                                    vistAImagingStudy.ErrorMessage =
                                        "This VistA image could not be converted into a format that can be included in a PDF. It WILL NOT be included in the RefDoc package.";
                                    vistAImagingStudy.ExceptionMessage = e.Message;
                                }
                            }
                        }

                        patientProfileViewModel.PatientProfile.VistAImagingStudies.Add(vistAImagingStudy);
                    }
                }
            }

            if (VistAImagingTotalTimelogging)
            {
                totalTime.Stop();
                Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception("Total elapsed time to retrieve VistA images: " + totalTime.Elapsed));
            }
        }

        public string getRandomICN()
        {
            int index;
            string selectedICN = string.Empty;
            string path = HttpContext.Current.Server.MapPath("~/App_Data/TestICNS.xml");
            XDocument xDoc = XDocument.Load(path);
            var ICNS = from icn in xDoc.Descendants("ICN")
                       select (string)icn.Value;

            string[] ICNSArray = ICNS.ToArray();

            // return a random icn from this array for test environment only
            Random rand = new Random();
            if (ICNSArray.Length > 0)
            {
                index = rand.Next(0, ICNSArray.Length - 1);
                selectedICN = ICNSArray[index];
            }

            return selectedICN;
        }

        private static bool IsValidImageOrPdf(VistAImagingFile vistAFile)
        {
            if (!vistAFile.FileType.Contains("pdf"))
            {
                try
                {
                    using (MemoryStream ms = new MemoryStream(vistAFile.FileBytes))
                    {
                        System.Drawing.Image.FromStream(ms);
                    }
                }
                catch (ArgumentException)
                {
                    return false;
                }
            }
            else
            {
                try
                {
                    using (MemoryStream ms = new MemoryStream(vistAFile.FileBytes))
                    {
                        var pdfReader = new PdfReader(ms);
                    }
                }
                catch
                {
                    return false;
                }
            }
            return true;
        }

        private static string GetRestrictedFileTypeUri(string uri, bool isPdf = false)
        {
            try
            {
                if (isPdf)
                {
                    return uri.Substring(0, uri.IndexOf("contentType=")) + "contentType=application/pdf,image/png,image/jpeg,image/gif,image/bmp";
                }
                else
                {
                    return uri.Substring(0, uri.IndexOf("contentType=")) + "contentType=image/png,image/jpeg,image/gif,image/bmp,application/pdf";
                }
            }
            catch
            {
                return null;
            }
        }
    }
}